/******************************************************************/
/* Title:      ICD-9-CM SURGERY FLAGS LOAD SOFTWARE, VERSION 1    */
/*                                                                */
/* Program:    ICD9_SURGERY_FLAGS_LOAD_PROGRAM.SAS                */
/*                                                                */
/* Description: This SAS program creates surgery flags for each   */
/*              ICD-9-CM procedure on a discharge record.  Values */
/*              of '1'=BROAD, '2'=NARROW, and '0'=NEITHER will    */
/*              populate each flag. 	                          */
/*                                                                */
/*              Please note that in order to build a list of all  */
/*              discharges with "BROAD" surgeries, analysts will  */
/*              need to capture codes with surgery flag=1 (BROAD) */
/*              as well as codes that have surgery flag=2 (NARROW)*/
/*              because the narrow codes are also broad by        */
/*              definition. See sample code in lines 73-82.       */
/*                                                                */
/*              The program is intended for data with ICD-9-CM    */
/*              procedure codes. The tool file consists only of   */
/*              ICD-9-CM codes considered surgical. These were    */
/*              selected from Procedure Classes 2,3, or 4. Any    */
/*              ICD-9-CM codes that are not in the tool files are */
/*              assigned a surgery flag of 0. The tool  file      */
/*              includes a subset of ICD-9-CM codes that were     */
/*              reviewed for the purposes of this flag as         */
/*              explained in the documentation.                   */
/*                                                                */
/*              The program assumes the use of discharge-level    */
/*              data where a single discharge record contains all */
/*              of the procedure codes on a single line.          */
/*                                                                */
/*              Users must indicate the maximum number of         */
/*              ICD-9-CM procedure codes found on their records.  */
/*              For example, if there are four ICD-9-CM codes on  */
/*              each record, the maximum number of ICD-9-CM codes */
/*              flag (NUMI9PRS) is set to 4 below. If there are   */
/*              no ICD-9-CM codes on any record, NUMI9PRS is set  */
/*              to 0 below.                                       */
/*              See lines 89-90 below to update.                  */
/*                                                                */
/*              There are two general sections to this program:   */
/*                                                                */
/*              1) The first section creates a temporary SAS      */
/*                 format using the ICD-9-CM surgery flag ".csv"  */
/*                 file. This format is used in step two to       */
/*                 create the surgery flag variables.             */
/*                                                                */
/*              2) The second section loops through the ICD-9-CM  */
/*                 procedure array on your SAS dataset and        */
/*                 creates a set of surgery flags from the        */
/*                 procedures present.                            */ 
/*                                                                */
/*                 Users should update lines 89-90,95-97,145,     */
/*                 155-156,173.                                   */
/*                                                                */
/* Optional Flags: Users may wish to create a record-level flag   */
/*                 to identify the entire discharge as surgical or*/
/*                 not. In order to do this, analysts should scan */
/*                 the ICD-9-CM surgery flags created by          */
/*                 the program below. If any surgery flag on a    */
/*                 record contains the values '1' or '2', two     */
/*                 record-level surgery flags would be created    */
/*                 with a possible value of 0 for not surgical    */
/*                 or 1 for surgical. If users would like a       */
/*                 record-level flag of surgical procedures,      */
/*                 broad_surgery=1 can be used since it           */
/*                 fundamentally includes both broad and narrow   */
/*                 surgical procedures.                           */
/*                                                                */
/*                 An example of SAS code to do this which could  */
/*                 be added at the end of the program follows:    */
/*                                                                */
/*                 NARROW_SURGERY = 0;                            */
/*                 BROAD_SURGERY = 0;                             */
/*                 %if &NUMI9PRS > 0 %then %do;                   */
/*                    DO I = 1 TO &NUMI9PRS;                      */
/*                       IF I9SFLAGS(I) ='2' THEN                 */
/*                       NARROW_SURGERY = 1;                      */
/*                       IF I9SFLAGS(I) IN('1','2') THEN          */
/*                       BROAD_SURGERY = 1;                       */
/*                    END;                                        */
/*                 %end;                                          */
/*                                                                */
/******************************************************************/

***************************************************************;
*** THE SAS MACRO FLAGS BELOW MUST BE UPDATED BY THE USER *****;
***************************************************************;
* SPECIFY THE NUMBER OF ICD-9-CM PROCEDURES ON YOUR RECORDS; %LET NUMI9PRS=5;
* SPECIFY YOUR SAS FILE NAME;                                %LET CORE=YOURSASDATA;

***********************************************************************************;
*** THE SAS PATHS BELOW MUST BE UPDATED BY THE USER TO MATCH YOUR FILE SYSTEM *****;
***********************************************************************************;
FILENAME INRAW1  'C:\TOOLS\SF\SURGERY_FLAGS_I9_2015.CSV'  LRECL=300;  
LIBNAME  IN1     'C:\SASDATA\';                 * Location of input discharge data;
LIBNAME  OUT1    'C:\SASDATA\';                 * Location of output data;



TITLE1 'CREATE SURGERY FLAGS';
TITLE2 'USE WITH DISCHARGE ADMINISTRATIVE DATA CONTAINING ICD-9-CM PROCECDURE CODES';

options mprint symbolgen;

%Macro surgery_flags;
/******************* SECTION 1: CREATE INFORMATS *******************/
/*  SAS Load the ICD-9-CM surgery flag tool and convert it into a  */
/*  temporary SAS format. The format will be used in the data step */
/*  below to assign the surgery flags for each  procedure on a     */
/*  record.                                                        */
/*******************************************************************/

DATA I9_SURGERY_FLAGS;
   INFILE INRAW1 DSD DLM=',' END = EOF FIRSTOBS=3;
	INPUT
	   START       : $CHAR4.
	   LABEL       : $1.
	;
   RETAIN HLO " ";
   FMTNAME = "$I9sf" ;
   TYPE    = "C" ;
   OUTPUT;

   IF EOF THEN DO ;
      START = " " ;
      LABEL = "0" ;
      HLO   = "O";
      OUTPUT ;
   END ;
RUN;

PROC FORMAT LIB=WORK CNTLIN = I9_SURGERY_FLAGS;
RUN;


/********** SECTION 2: CREATE SURGERY FLAGS VARIABLES *************/
/*  Create surgery flag variables for each ICD-9-CM code using    */
/*  the SAS format created above and the SAS file you wish to     */
/*  augment. Users can change the names of the output             */
/*  surgery flag variables if needed here. It is also             */
/*  important to make sure that the correct ICD-9 procedure names */
/*  from your SAS file are used in the array 'I9PRS'.             */
/******************************************************************/  
DATA OUT1.NEW_SURGERY_FLAGS (DROP = i);			    * The SAS output dataset name can be changed here;
  SET IN1.&CORE;

  /***************************************************/
  /*  First build an array for all of the surgery    */
  /*  flag variables that are going to be created.   */
  /*  Also build an array that contains the existing */
  /*  I9 codes on the input file.                    */
  /***************************************************/
  %if &NUMI9PRS > 0 %then %do;
     ARRAY I9SFLAGS (*)     $1 SURG_ICD9_FLG1-SURG_ICD9_FLG&NUMI9PRS;  * Suggested name for the I-9 minor therapeutic and diagnostic surgery flags variables;
     ARRAY I9PRS    (*)     $4 PR1-PR&NUMI9PRS;                        * Change I9 procedure variable names to match your file;
  %end;
 
  /***************************************************/
  /*  Loop through the ICD-9-CM procedure codes      */ 
  /*  on your SAS input dataset and create a surgery */
  /*  flag variable for each code. This step makes   */
  /*  use of the I9 arrays above.                    */
  /***************************************************/
  %if &NUMI9PRS > 0 %then %do;
     DO I = 1 TO &NUMI9PRS;	     
	     I9SFLAGS(I) = PUT(I9PRS(I),$I9sf.);
     END;  
  %end;

RUN;

PROC PRINT DATA=OUT1.NEW_SURGERY_FLAGS   (OBS=10);  * If the SAS output dataset name changed above, make sure to change it here;
   VAR %if &NUMI9PRS > 0 %then %do;PR1 SURG_ICD9_FLG1%end;
     		 ;
  title3 "PARTIAL PRINT OF THE SURGERY FLAGS FILE";
RUN;
%Mend surgery_flags;
%surgery_flags;



